/*++

	Copyright (c) 1997,  Dr. Johannes Heidenhain GmbH

	Module Name:	IK121Drv.C
	

	Abstract:		Windows NT kernel-mode device driver for IK 121

	Notes:			This kernel driver allows Win32 applications to access
					IK 121 hardware through DeviceIOControl.


	History:		Version command added				Schroll E.		11.06.99
--*/



#include "ntddk.h"
#include "string.h"
#include "..\Include\DrvFunc.h"



#ifdef _DEBUG
	const char DrvVers[] = "DRV V1.1(D)";		// 11.06.1999
#else											   
	const char DrvVers[] = "DRV V1.1(R)";
#endif


#ifdef DBG
  #define kdPrint(arg)	DbgPrint arg
  #define kdBreak		DbgBreakPoint
#else
  #define kdPrint(arg)
  #define kdBreak
#endif




typedef struct _DEVICE_EXTENSION
{	PDEVICE_OBJECT	DeviceObject;
	ULONG			DeviceType;
	PULONG			PortBase[8];
	ULONG			DevStat;
} DEVICE_EXTENSION, *PDEVICE_EXTENSION;




// Function prototypes
VOID		IK121DrvUnload	 ( IN PDRIVER_OBJECT	DriverObject );
NTSTATUS	IK121DrvDispatch ( IN PDEVICE_OBJECT	DeviceObject,
							   IN PIRP				Irp );


VOID IK121WriteErrorLog		 ( IN PDEVICE_OBJECT	DeviceObject, 
							   IN ULONG				EventID, 
							   IN ULONG				ErrorValue ); 




NTSTATUS DriverEntry ( IN PDRIVER_OBJECT  DriverObject,		// Pointer to driver object 
					   IN PUNICODE_STRING RegistryPath)		// Pointer to registry path


{

	PDEVICE_OBJECT		deviceObject;						// Pointer to device object
	PDEVICE_EXTENSION	deviceExtension;					// Pointer to device extension
	NTSTATUS			ntStatus;							// NT-Status
	WCHAR nameText[] = L"\\Device\\IK121Drv";				// Name of device driver
	WCHAR linkText[] = L"\\DosDevices\\IK121Drv";			// DOS-Name of device driver
	UNICODE_STRING		nameString, linkString;				// Device name

	PHYSICAL_ADDRESS	PortAddress[8], MappedAddress;		// Port address of IK 121
	ULONG				MemType  = 1;						// ISA bus
	ULONG				PortSize = 16;						// Card I/O space size
	ULONG				index;								// Card index
	ULONG				NumberOfCards = 0;					// Number of registered cards

	PCM_RESOURCE_LIST	ResourceList	   = NULL;			// Pointer to resource usage list to report to system
	CM_RESOURCE_LIST	EmptyResourceList;		 			// Empty resource usage list
	ULONG				SizeOfResourceList = 0;				// Size of resource list
	PCM_FULL_RESOURCE_DESCRIPTOR	PFullResDesc;			// Full resource descriptor
	PCM_PARTIAL_RESOURCE_DESCRIPTOR	PPartResDesc;			// Partial resource descriptor
	BOOLEAN				ResourceConflict   = TRUE;			// Resoruce conflict

	ULONG				DefaultBase = 0x330;				// Default base address IK 121
	ULONG				Zero		= 0x0;
	RTL_QUERY_REGISTRY_TABLE	paramTable[9];				// Parameter read from registry
	UNICODE_STRING				parameters;
	UNICODE_STRING				paramPath;




	kdPrint (( "IK121Drv: DriverEntry started\n" ));



	// Create device
	////////////////

	RtlInitUnicodeString ( &nameString, nameText );

	ntStatus = IoCreateDevice ( DriverObject,				// Pointer to driver object
								sizeof(DEVICE_EXTENSION),	// Size of device extension
								&nameString,				// Device name
								FILE_DEVICE_IK121,			// Device type
								0,							// Device characteristics
								TRUE,						// Exclusive device
								&deviceObject );			// Newly created device object
								
 
	if (!NT_SUCCESS (ntStatus))
	{
		IK121WriteErrorLog (deviceObject, 0 ,ntStatus);
		kdPrint (( "IK121Drv: Can't create device\n" ));
		return ntStatus;
	}


	// Fill device object
	/////////////////////

	deviceObject->Flags |= DO_BUFFERED_IO;

	deviceExtension = (PDEVICE_EXTENSION) deviceObject->DeviceExtension;
	RtlZeroMemory ( deviceExtension, sizeof(*deviceExtension) );
	deviceExtension->DeviceObject = deviceObject;
	deviceExtension->DeviceType   = FILE_DEVICE_IK121;



	// Create symbolic link
	///////////////////////

	RtlInitUnicodeString ( &linkString, linkText );
	ntStatus = IoCreateSymbolicLink  ( &linkString, &nameString );

	if (!NT_SUCCESS (ntStatus))
	{
		IK121WriteErrorLog (deviceObject, 3 ,ntStatus);
		kdPrint (( "IK121Drv: Can't create symbolic link\n" ));
		IoDeleteDevice (DriverObject->DeviceObject);
		return ntStatus;
	}


	DriverObject->DriverUnload = IK121DrvUnload;							// Driver unload 
	DriverObject->MajorFunction[IRP_MJ_CREATE] =
	DriverObject->MajorFunction[IRP_MJ_CLOSE]  =
	DriverObject->MajorFunction[IRP_MJ_READ]   =
	DriverObject->MajorFunction[IRP_MJ_WRITE] =
	DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = IK121DrvDispatch;	// Driver dispatch 




	// Read port address from registry
	//////////////////////////////////

    RtlInitUnicodeString(&parameters, L"\\Parameters");
    paramPath.Length = 0;
    paramPath.MaximumLength = RegistryPath->Length + parameters.Length + sizeof(WCHAR);
    paramPath.Buffer = ExAllocatePool(PagedPool, paramPath.MaximumLength);
   
	if (paramPath.Buffer != NULL)
	{
	    RtlZeroMemory(paramPath.Buffer, paramPath.MaximumLength);
		RtlAppendUnicodeStringToString(&paramPath, RegistryPath);
	    RtlAppendUnicodeStringToString(&paramPath, &parameters);

		RtlZeroMemory (paramTable,  sizeof (paramTable));
		RtlZeroMemory (PortAddress, sizeof (PortAddress));

	    paramTable[0].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[0].Name			= L"IK_Base_1";
		paramTable[0].EntryContext	= &PortAddress[0].LowPart;
	    paramTable[0].DefaultType	= REG_DWORD;
		paramTable[0].DefaultData	= &DefaultBase;
	    paramTable[0].DefaultLength = sizeof(ULONG);

	    paramTable[1].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[1].Name			= L"IK_Base_2";
		paramTable[1].EntryContext	= &PortAddress[1].LowPart;
	    paramTable[1].DefaultType	= REG_DWORD;
		paramTable[1].DefaultData	= &Zero;
	    paramTable[1].DefaultLength = sizeof(ULONG);

	    paramTable[2].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[2].Name			= L"IK_Base_3";
		paramTable[2].EntryContext	= &PortAddress[2].LowPart;
	    paramTable[2].DefaultType	= REG_DWORD;
		paramTable[2].DefaultData	= &Zero;
	    paramTable[2].DefaultLength = sizeof(ULONG);

	    paramTable[3].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[3].Name			= L"IK_Base_4";
		paramTable[3].EntryContext	= &PortAddress[3].LowPart;
	    paramTable[3].DefaultType	= REG_DWORD;
		paramTable[3].DefaultData	= &Zero;
	    paramTable[3].DefaultLength = sizeof(ULONG);

	    paramTable[4].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[4].Name			= L"IK_Base_5";
		paramTable[4].EntryContext	= &PortAddress[4].LowPart;
	    paramTable[4].DefaultType	= REG_DWORD;
		paramTable[4].DefaultData	= &Zero;
	    paramTable[4].DefaultLength = sizeof(ULONG);

		paramTable[5].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[5].Name			= L"IK_Base_6";
		paramTable[5].EntryContext	= &PortAddress[5].LowPart;
	    paramTable[5].DefaultType	= REG_DWORD;
		paramTable[5].DefaultData	= &Zero;
	    paramTable[5].DefaultLength = sizeof(ULONG);

	    paramTable[6].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[6].Name			= L"IK_Base_7";
		paramTable[6].EntryContext	= &PortAddress[6].LowPart;
	    paramTable[6].DefaultType	= REG_DWORD;
		paramTable[6].DefaultData	= &Zero;
	    paramTable[6].DefaultLength = sizeof(ULONG);

	    paramTable[7].Flags = RTL_QUERY_REGISTRY_DIRECT;
		paramTable[7].Name			= L"IK_Base_8";
		paramTable[7].EntryContext	= &PortAddress[7].LowPart;
	    paramTable[7].DefaultType	= REG_DWORD;
		paramTable[7].DefaultData	= &Zero;
	    paramTable[7].DefaultLength = sizeof(ULONG);


		ntStatus = RtlQueryRegistryValues( RTL_REGISTRY_ABSOLUTE | RTL_REGISTRY_OPTIONAL,
										   paramPath.Buffer,
										   paramTable,
										   NULL, NULL);

		if (!NT_SUCCESS (ntStatus))
		{
			IK121WriteErrorLog (deviceObject, 1 , ntStatus);
			PortAddress[0].LowPart = DefaultBase;		// Set default I/O address for first IK 121
		}
	    ExFreePool(paramPath.Buffer);
	}
	else
	{
		IK121WriteErrorLog (deviceObject, 2 , 0);
		PortAddress[0].LowPart  = DefaultBase;			// Set default I/O address for first IK 121

	}




	// Report resource usage to system
	//////////////////////////////////

	kdPrint (( "IK121Drv: Report resource usage\n" ));;


	SizeOfResourceList = sizeof(CM_RESOURCE_LIST);
	for ( index = 0; index < 8; index++ )
		if (PortAddress[index].LowPart != 0)
		{
			SizeOfResourceList = SizeOfResourceList + sizeof(CM_PARTIAL_RESOURCE_DESCRIPTOR);
			NumberOfCards++;
		}


	kdPrint (( "IK121Drv: Claiming resources for %d card(s)\n", NumberOfCards ));;

	ResourceList = ExAllocatePool ( PagedPool, SizeOfResourceList);
	

	if (ResourceList != NULL)
	{

		RtlZeroMemory ( ResourceList, SizeOfResourceList);

		ResourceList->Count = 1;

		PFullResDesc = &ResourceList->List[0];

		PFullResDesc->InterfaceType					= Isa;
		PFullResDesc->PartialResourceList.Count		= NumberOfCards;
		PFullResDesc->PartialResourceList.Version	= 0;
		PFullResDesc->PartialResourceList.Revision	= 0;

		PPartResDesc = &(PFullResDesc->PartialResourceList.PartialDescriptors[0]);


		for ( index = 0; index < 8; index++ ) 
			if (PortAddress[index].LowPart != 0)
			{
				PPartResDesc->Type				= CmResourceTypePort;
				PPartResDesc->ShareDisposition	= CmResourceShareDriverExclusive;
				PPartResDesc->Flags				= CM_RESOURCE_PORT_IO;
				PPartResDesc->u.Port.Start		= PortAddress[index];
				PPartResDesc->u.Port.Length		= PortSize;
				PPartResDesc++;

				kdPrint (( "IK121Drv: Reporting resource %d at 0x%08lX\n", index, PortAddress[index] ));
			}

		ntStatus = IoReportResourceUsage ( NULL,
										   DriverObject,
										   ResourceList,
										   SizeOfResourceList,
										   NULL,
										   NULL,
										   0,
										   FALSE,
										   &ResourceConflict );

		ExFreePool (ResourceList);


		if (!NT_SUCCESS (ntStatus))	
		{
			IK121WriteErrorLog (deviceObject, 4 , ntStatus);
			kdPrint (( "IK121Drv: Resource conflict: Status = 0x%08lX\n", ntStatus));

			RtlInitUnicodeString ( &linkString, linkText );
			IoDeleteSymbolicLink ( &linkString );
			IoDeleteDevice (DriverObject->DeviceObject);
			return ntStatus;
		}
	}
	else
	{		
		IK121WriteErrorLog (deviceObject, 2 , 0);
		kdPrint (( "IK121Drv: Can't allocate memory for resource list\n" ));

	 	RtlInitUnicodeString ( &linkString, linkText );
	 	IoDeleteSymbolicLink ( &linkString );
	 	IoDeleteDevice (DriverObject->DeviceObject);
	 	return STATUS_UNSUCCESSFUL;
	}



	kdPrint (( "IK121Drv: Get hardware access\n" ));;


	// Translate physical address to virtual address
	////////////////////////////////////////////////

	for ( index = 0; index < 8; index++ ) 
		if (PortAddress[index].LowPart != 0)
		{
			ntStatus = HalTranslateBusAddress ( Isa,
												0,
												PortAddress[index],
												&MemType,
												&MappedAddress );


			if (!NT_SUCCESS (ntStatus))
			{
				IK121WriteErrorLog (deviceObject, 5 ,ntStatus);
				kdPrint (( "IK121Drv: Hardware access failed\n" ));

				IoReportResourceUsage ( NULL,
										DriverObject,
										&EmptyResourceList,
										sizeof(EmptyResourceList),
										NULL,
										NULL,
										0,
										FALSE,
										&ResourceConflict );



				RtlInitUnicodeString ( &linkString, linkText );
				IoDeleteSymbolicLink ( &linkString );

				IoDeleteDevice (DriverObject->DeviceObject);
				return ntStatus;
			}
			deviceExtension->PortBase[index] = (PVOID)MappedAddress.LowPart;
		}
		else deviceExtension->PortBase[index] = NULL;


	kdPrint (( "IK121Drv: DriverEntry ends\n" ));;
	return ntStatus;
}





VOID IK121DrvUnload ( IN PDRIVER_OBJECT	DriverObject )

{
	WCHAR linkText[] = L"\\DosDevices\\IK121Drv";
	UNICODE_STRING	linkString;								// Device name

	CM_RESOURCE_LIST	EmptyResourceList;					// Empty resource usage list
	BOOLEAN				ResourceConflict;					// Resource already in use


	
	kdPrint (( "IK121Drv: Driver unload\n" ));


	// Unload driver: remove resource usage, delete symbolic link, delete device
	////////////////////////////////////////////////////////////////////////////

	RtlZeroMemory (&EmptyResourceList, sizeof(EmptyResourceList) );

	IoReportResourceUsage ( NULL,
							DriverObject,
							&EmptyResourceList,
							sizeof(EmptyResourceList),
							NULL,
							NULL,
							0,
							FALSE,
							&ResourceConflict );


	RtlInitUnicodeString ( &linkString, linkText );
	IoDeleteSymbolicLink ( &linkString );

	IoDeleteDevice (DriverObject->DeviceObject);
}



NTSTATUS IK121DrvDispatch ( IN PDEVICE_OBJECT	deviceObject,
							IN PIRP				Irp )
{
	NTSTATUS			ntStatus = STATUS_SUCCESS;				// NT-Status
	PIO_STACK_LOCATION	irpStack;
	ULONG				ioControlCode;
	ULONG				i;										// Loop counter
	ULONG				bytesToRead, bytesToWrite;				// Number of bytes for i/o
	PUCHAR				irpBufferB;								// Byte  buffer pointer for i/o
	PUSHORT				irpBufferS;								// Short buffer pointer for i/o
	PULONG				irpBufferL;								// Long  buffer pointer for i/o
	USHORT				adrSeg, adrOfs, adrSel, cardNum;		// Address latch, offset and select
    USHORT				IKStat; 
	PDEVICE_EXTENSION	deviceExtension = deviceObject->DeviceExtension;

	irpStack = IoGetCurrentIrpStackLocation(Irp);
	switch (irpStack->MajorFunction)
	{
	case IRP_MJ_CREATE:
		kdPrint (( "IK121Drv: Dispatched to create\n" ));
		Irp->IoStatus.Information = 0;
		break;

	case IRP_MJ_CLOSE:
		kdPrint (( "IK121Drv: Dispatched to close\n" ));
		deviceExtension->DevStat  = 0;
		Irp->IoStatus.Information = 0;
		break;

	case IRP_MJ_READ:
		kdPrint (( "IK121Drv: Dispatched to read\n" ));
		Irp->IoStatus.Information = 0;
		break;

	case IRP_MJ_WRITE:
		kdPrint (( "IK121Drv: Dispatched to write\n" ));
		Irp->IoStatus.Information = 0;
		break;

	case IRP_MJ_DEVICE_CONTROL:
		kdPrint (( "IK121Drv: DeviceControl - " ));

		ioControlCode = irpStack->Parameters.DeviceIoControl.IoControlCode;
		switch (ioControlCode)
		{
		case IOCTL_INPUT:

			bytesToRead = irpStack->Parameters.DeviceIoControl.OutputBufferLength;
			irpBufferS = Irp->AssociatedIrp.SystemBuffer;
			irpBufferL = Irp->AssociatedIrp.SystemBuffer;
			
			cardNum=   *irpBufferS >> 1;				// Axis number --> Card number (2 axes on one card)
			adrSeg = (*(irpBufferS+1) & 0x1C) >> 2;		// Reg  number --> Page number
			adrOfs = ( *irpBufferS & 0x01) << 2;		// Axis number --> Axis select
			adrSel = (*(irpBufferS+1) & 0x03);			// Reg  number --> Reg  select


			kdPrint (( "Reading card 0x%X axis %d register 0x%02X: ",deviceExtension->PortBase[cardNum], (*irpBufferS & 0x1), *(irpBufferS+1) ));

			if (deviceExtension->PortBase[cardNum] != NULL)
			{
				WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum] | 8), adrSeg );

				switch (bytesToRead)
				{
					case 2:
					{
						*irpBufferS = READ_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
																		| adrOfs | adrSel ));
						Irp->IoStatus.Information = bytesToRead;
						kdPrint (( "0x%04X\n",*irpBufferS ));
						break;
					}


					case 4:
					{

						if (adrSel==0)	// Read longword from quad address --> same page
						{
							*irpBufferL = READ_PORT_ULONG ( (PULONG) ( (ULONG)deviceExtension->PortBase[cardNum]
																			| adrOfs | adrSel ));
						}
						else	// Read longword from different pages 
						{
							*(irpBufferS) = READ_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
																			| adrOfs | adrSel ));
							adrSeg=(adrSeg+1) & 0x07; 	// Next page number !!!
							adrSel=(adrSel+2) & 0x03; 	// Next register address !!!
							WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum] | 8), adrSeg );
							*(irpBufferS+1) = READ_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
							   												      | adrOfs | adrSel ));
						}
						Irp->IoStatus.Information = bytesToRead;
						kdPrint (( "0x%08lX\n",*irpBufferL ));
						break;
					}

					default:
					{
						kdPrint (( "Illegal byte count!\n" ));
						if ((deviceExtension->DevStat & 0x1) == 0)
						{
							IK121WriteErrorLog (deviceObject, 6, bytesToRead);
							deviceExtension->DevStat |= 0x1;
						}
						Irp->IoStatus.Information = 0;
						break;
					}
				}

			}
			else
			{
				kdPrint (( "Card not found!\n" ));
				if ((deviceExtension->DevStat & 0x2) == 0)
				{
					IK121WriteErrorLog (deviceObject, 7, cardNum);
					deviceExtension->DevStat |= 0x2;
				}
				Irp->IoStatus.Information = 0;
			}
		break;



		case IOCTL_OUTPUT:

			bytesToWrite = irpStack->Parameters.DeviceIoControl.InputBufferLength;
			irpBufferS = Irp->AssociatedIrp.SystemBuffer;

			cardNum=   *irpBufferS >> 1;				// Axis number --> Card number (2 axes on one card)
			adrSeg = (*(irpBufferS+1) & 0x1C) >> 2;		// Reg  number --> Page number
			adrOfs = ( *irpBufferS & 0x01) << 2;		// Axis number --> Axis select
			adrSel =  *(irpBufferS+1) & 0x03;			// Reg  number --> Reg  select


			kdPrint (( "Writing 0x%04X card 0x%X axis %d register 0x%02X: \n",*(irpBufferS+2), deviceExtension->PortBase[cardNum], (*irpBufferS & 0x1), *(irpBufferS+1) ));


			if (deviceExtension->PortBase[cardNum] != NULL)
			{
				WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
																| 8), adrSeg );
				WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
																| adrOfs | adrSel ), *(irpBufferS+2)); 
				Irp->IoStatus.Information = bytesToWrite;
			}
			else
			{
				kdPrint (( "\nCard not found!\n" ));
				if ((deviceExtension->DevStat & 0x2) == 0)
				{
					IK121WriteErrorLog (deviceObject, 7, cardNum);
					deviceExtension->DevStat |= 0x2;
				}
				Irp->IoStatus.Information = 0;
			}
			break;


		case IOCTL_STATUS:
			bytesToRead = irpStack->Parameters.DeviceIoControl.OutputBufferLength;
			irpBufferL = Irp->AssociatedIrp.SystemBuffer;
			kdPrint (( "IK cards at:" ));

			adrSeg =  0x1C >> 2;				// Reg  number --> Page number
			adrOfs =  0;						// Axis number --> Axis select
			adrSel =  0;						// Reg  number --> Reg  select

			for (cardNum=0; cardNum<8; cardNum++)
			{
				if (deviceExtension->PortBase[cardNum]!=0)
				{
					WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum] | 8), adrSeg );
					IKStat = READ_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
																	| adrOfs | adrSel ));

					if ( (IKStat & 0xFF00) == 0x0800 || (IKStat & 0xFF00) == 0x0900 )
					{
						*(irpBufferL+cardNum) = (ULONG)deviceExtension->PortBase[cardNum];
						kdPrint (( "(%d)=0x%X\n", cardNum, *(irpBufferL+cardNum) ));
					}
					else
					{
						IK121WriteErrorLog (deviceObject, 7, cardNum);
						*(irpBufferL+cardNum) = 0;
					}
				}
				else *(irpBufferL+cardNum) = 0;
			}
				Irp->IoStatus.Information = 8* sizeof(ULONG);

			break;


		case IOCTL_SETI2C:

			bytesToWrite = irpStack->Parameters.DeviceIoControl.InputBufferLength;
			irpBufferS = Irp->AssociatedIrp.SystemBuffer;

			cardNum=   *irpBufferS;		// Card number

			// Address SCL/SDA port = 0x12 !!
			adrSeg = 0x04;				// Page number I2C-Input
			adrOfs = 0x04;				// Axis 2
			adrSel = 0x02;				// Reg  select 0


		#ifdef DBG
			kdPrint (( "Set I2C at card 0x%X:  SDA=",  deviceExtension->PortBase[cardNum] ));
			if (*(irpBufferS+1) & 0x0800) kdPrint (( " |  " ));   else kdPrint (( "   |" ));
			kdPrint (( "  SCL=" ));
			if (*(irpBufferS+1) & 0x8000) kdPrint (( " |  \n" )); else kdPrint (( "   |\n" ));
		#endif
																	   

			if (deviceExtension->PortBase[cardNum] != NULL)
			{
				WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum] | 8), adrSeg );


				WRITE_PORT_USHORT ( (PUSHORT) ( (ULONG)deviceExtension->PortBase[cardNum]
																| adrOfs | adrSel ), *(irpBufferS+1)); 

				KeStallExecutionProcessor (10);		// Wait 10s
				Irp->IoStatus.Information = bytesToWrite;
			}
			else
			{
				kdPrint (( "\nCard not found!\n" ));
				if ((deviceExtension->DevStat & 0x2) == 0)
				{
					IK121WriteErrorLog (deviceObject, 7, cardNum);
					deviceExtension->DevStat |= 0x2;
				}
				Irp->IoStatus.Information = 0;
			}
			break;



		case IOCTL_VERSION:
			irpBufferB = Irp->AssociatedIrp.SystemBuffer;
				 
			
			for (i=0; i<sizeof(DrvVers); i++)
				*irpBufferB++ = DrvVers[i];

			Irp->IoStatus.Information = sizeof(DrvVers);
			break;





		default:
			if ((deviceExtension->DevStat & 0x4) == 0)
			{
				IK121WriteErrorLog (deviceObject, 9 ,ioControlCode);
				deviceExtension->DevStat |= 0x4;
			}	
			kdPrint (( "unknown minor function\n" ));
			ntStatus = STATUS_NOT_IMPLEMENTED;
			Irp->IoStatus.Information = 0;
			break;
		}
		break;



	default:
		if ((deviceExtension->DevStat & 0x4) == 0)
		{
			IK121WriteErrorLog (deviceObject, 8 ,0);
			deviceExtension->DevStat |= 0x4;
		}	
		kdPrint (( "IK121Drv: Dispatched to unknown major function\n" ));
		ntStatus = STATUS_NOT_IMPLEMENTED;
		Irp->IoStatus.Information = 0;
		break;
	}

	Irp->IoStatus.Status = ntStatus;
	IoCompleteRequest (Irp,IO_NO_INCREMENT);
	return ntStatus;
}




void IK121WriteErrorLog ( IN PDEVICE_OBJECT DeviceObject, 
				 	      IN ULONG EventID, 
					      IN ULONG ErrorValue  ) 
 
{ 
    PIO_ERROR_LOG_PACKET errorLogEntry; 
    UCHAR		 EntrySize; 
    PUCHAR		 StringLoc; 
    ULONG		 TempError; 
    static WCHAR ErrorBuffer[11]   = L"00000000"; 
    static WCHAR ErrorText[10][30] = { L"Can't create device ",			// EventID 0
							  		   L"Can't read registry values ",	// EventID 1:
									   L"Can't allocate memory ",		// EventID 2:
									   L"Can't create symbolic link ", 	// EventID 3:
									   L"Resource conflict ",		 	// EventID 4:
									   L"Hardware access failed ",		// EventID 5:
									   L"Illegal byte count ",			// EventID 6:
									   L"Card not found ",				// EventID 7:
									   L"Unknown major function ",		// EventID 8:
									   L"Unknown minor function "		// EventID 8:
									 };
    short i; 
 
    EntrySize = sizeof(IO_ERROR_LOG_PACKET) + sizeof(ErrorText[EventID]); 

	if (ErrorValue!=0)
	{
		// Convert the error value into a string
		TempError = ErrorValue; 
		for (i=9; i>=0; i--)
		{ 
			ErrorBuffer[i] = (WCHAR)((TempError % 10) + L'0'); 
	        TempError /= 10; 
		} 
 
		EntrySize += sizeof(ErrorBuffer);
	}


    errorLogEntry = (PIO_ERROR_LOG_PACKET)IoAllocateErrorLogEntry( DeviceObject, EntrySize );
 
    if (errorLogEntry != NULL)
	{ 
        errorLogEntry->MajorFunctionCode = (UCHAR)-1; 
        errorLogEntry->RetryCount = (UCHAR)-1; 
        errorLogEntry->DumpDataSize = 0;

		if (ErrorValue!=0) errorLogEntry->NumberOfStrings = 2; 
        else errorLogEntry->NumberOfStrings = 1; 
		
		errorLogEntry->StringOffset = sizeof(IO_ERROR_LOG_PACKET);
        errorLogEntry->EventCategory = 0; 
        errorLogEntry->ErrorCode =  EventID;
        errorLogEntry->UniqueErrorValue = ErrorValue; 
        errorLogEntry->FinalStatus = 0;
        errorLogEntry->SequenceNumber = (ULONG)-1; 								
        errorLogEntry->IoControlCode = 0;
		errorLogEntry->DeviceOffset.QuadPart = 0;

        StringLoc = ((PUCHAR)errorLogEntry) + errorLogEntry->StringOffset; 
        RtlCopyMemory (StringLoc, ErrorText[EventID], sizeof(ErrorText[EventID])); 


		if (ErrorValue!=0)
		{
			while (*StringLoc) StringLoc+=sizeof(WCHAR);
			StringLoc+=sizeof(WCHAR);
			RtlCopyMemory (StringLoc, ErrorBuffer, sizeof(ErrorBuffer)); 
		}

        IoWriteErrorLogEntry(errorLogEntry); 
    } 
 
}
 
